package furny.ga.operators;

import furny.entities.Furniture;
import furny.furndb.FurnCache;
import furny.ga.FurnEntry;
import furny.ga.FurnLayoutIndividual;
import furny.ga.PseudoSpace;
import furny.ga.RoomVector;
import furny.ga.util.FurnLayoutIOUtil;
import furny.ga.util.FurnitureUtil;
import ga.core.GA;
import ga.core.goperators.IMutationOp;
import ga.core.goperators.ProbabilityOp;
import ga.core.validation.GAContext;
import ga.view.interfaces.IPhenotypeSpace;

import java.util.List;

/**
 * The real mutation operator mutates genes randomly. Every single gene has a
 * mutation chance given by the mutation probability.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class RealMutationOp extends ProbabilityOp implements
    IMutationOp<FurnLayoutIndividual> {

  private final int pNewFurniture;
  private final int pRemoveFurniture;

  private int wID = 6;
  private int wTranslate = 40;
  private int wRotate = 2;

  /**
   * Creates the operator with the given mutation probability and given
   * probabilities to add or remove furniture segments. The widths are also
   * defined.
   * 
   * @param pMutate
   *          Mutation probability.
   * @param wID
   *          ID mutation width.
   * @param wTranslate
   *          Translation mutation width.
   * @param wRotate
   *          Rotation mutation width.
   * @param pNewFurniture
   *          Probability to add a furniture segment.
   * @param pRemoveFurniture
   *          Probability to remove a furniture segment.
   * 
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public RealMutationOp(final int pMutate, final int wID, final int wTranslate,
      final int wRotate, final int pNewFurniture, final int pRemoveFurniture) {
    this(pMutate);
    this.wID = wID;
    this.wTranslate = wTranslate;
    this.wRotate = wRotate;
  }

  /**
   * Creates the operator with the given mutation probability and given
   * probabilities to add or remove furniture segments.
   * 
   * @param pMutate
   *          Mutation probability.
   * @param pNewFurniture
   *          Probability to add a furniture segment.
   * @param pRemoveFurniture
   *          Probability to remove a furniture segment.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public RealMutationOp(final int pMutate, final int pNewFurniture,
      final int pRemoveFurniture) {
    super(pMutate);
    this.pNewFurniture = pNewFurniture;
    this.pRemoveFurniture = pRemoveFurniture;
  }

  /**
   * Creates the operator with the given mutation probability.
   * 
   * @param pMutate
   *          Mutation probability.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public RealMutationOp(final int pMutate) {
    this(pMutate, 0, 0);
  }

  @Override
  public FurnLayoutIndividual mutate(final FurnLayoutIndividual individual,
      final GAContext context) {
    final FurnLayoutIndividual newInd = individual.clone();

    int genomeMinLength = 1;
    int genomeMaxLength = 6;

    if (context != null) {
      if (context.containsKey(GA.KEY_GENOME_MIN_LENGTH)) {
        genomeMinLength = (Integer) context.get(GA.KEY_GENOME_MIN_LENGTH);
      }

      if (context.containsKey(GA.KEY_GENOME_MAX_LENGTH)) {
        genomeMaxLength = (Integer) context.get(GA.KEY_GENOME_MAX_LENGTH);
      }
    }

    if (doOperate(pRemoveFurniture)
        && newInd.getFurnitures().size() > genomeMinLength) {
      newInd.getFurnitures().remove(
          getRandom().nextInt(newInd.getFurnitures().size()));
    }

    for (final FurnEntry entry : newInd.getFurnitures()) {
      if (doOperate()) {
        // mutate the ID
        // TODO
        entry.setFurniture(FurnitureUtil.getOtherFurnitureBySimilarityDistance(
            entry.getFurniture(), getRandom().nextInt(wID) + 1));
      }

      // mutate the translation and rotation
      RoomVector vec = entry.getVector();

      // mutation on all genes with pMut
      if (doOperate()) {
        // mutate x
        final int translation = getRandom().nextInt(wTranslate + 1)
            - (wTranslate / 2);

        vec = vec.getTranslatedInstance(translation, 0);
      }

      if (doOperate()) {
        // mutate y
        final int translation = getRandom().nextInt(wTranslate + 1)
            - (wTranslate / 2);

        vec = vec.getTranslatedInstance(0, translation);
      }

      if (doOperate()) {
        // mutate rot

        // see if it should rotate
        int dRot = getRandom().nextInt(wRotate);
        // random invert the sign
        if (dRot != 0 && getRandom().nextBoolean()) {
          dRot *= -1;
        }

        vec = vec.getRotatedInstance(dRot);
      }

      entry.setVector(vec);
    }

    if (doOperate(pNewFurniture)
        && newInd.getFurnitures().size() < genomeMaxLength) {

      // fallback values
      float minX = -10f;
      float maxX = 10f;
      float minY = -10f;
      float maxY = 10f;

      IPhenotypeSpace space = null;

      if (context != null) {
        final Object o = context.get(GA.KEY_VALIDATION_SPACE);

        if (o != null && o instanceof IPhenotypeSpace) {
          space = (IPhenotypeSpace) o;

          minX = (float) space.getOutterBounds().getX();
          maxX = (float) (space.getOutterBounds().getX() + space
              .getOutterBounds().getWidth());

          minY = (float) space.getOutterBounds().getY();
          maxY = (float) (space.getOutterBounds().getY() + space
              .getOutterBounds().getHeight());
        }
      }

      final List<Furniture> all = FurnCache.getInstance().getAllFurnitures();

      final int x = Math
          .round(((getRandom().nextFloat() * (maxX - minX)) + minX) * 100);
      final int y = Math
          .round(((getRandom().nextFloat() * (maxY - minY)) + minY) * 100);
      final int rotationSteps = getRandom().nextInt(4);

      final Furniture f = all.get(getRandom().nextInt(all.size()));

      newInd.getFurnitures().add(
          new FurnEntry(new RoomVector(x, y, rotationSteps), f));

    }

    return newInd;
  }

  /**
   * Main method for testing.
   * 
   * @param args
   *          No arguments required.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    final GAContext context = new GAContext();
    context.put(GA.KEY_VALIDATION_SPACE, new PseudoSpace(10f, 10f));

    final FurnLayoutIndividual ind = new FurnLayoutIndividual(context);
    ind.initRandomly();

    final RealMutationOp crossover = new RealMutationOp(100);

    System.out.println("\n BEFORE\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(ind
        .getSimpleGenotype()));

    final FurnLayoutIndividual newInd = crossover.mutate(ind, context);

    System.out.println("\n AFTER\n");
    System.out.println(FurnLayoutIOUtil.printSimpleGenotype(newInd
        .getSimpleGenotype()));
  }
}
